package gov.va.med.mhv.usermgmt.service.impl;

import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.enums.Enum;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.tigris.atlas.messages.Message;
import org.tigris.atlas.messages.MessageUtils;
import org.tigris.atlas.messages.Messages;
import org.tigris.atlas.service.BooleanServiceResponse;
import org.tigris.atlas.service.EntityServiceResponse;
import org.tigris.atlas.service.ServiceResponse;
import org.tigris.atlas.service.StringServiceResponse;
import org.tigris.atlas.service.VoidServiceResponse;
import org.tigris.atlas.util.CalendarUtils;

import gov.va.med.mhv.core.util.DescriptionBuilder;
import gov.va.med.mhv.core.util.MessagesStringBuilder;
import gov.va.med.mhv.core.util.Precondition;
import gov.va.med.mhv.core.util.TimestampUtils;
import gov.va.med.mhv.integration.registry.transfer.Gender;
import gov.va.med.mhv.integration.registry.transfer.Name;
import gov.va.med.mhv.integration.registry.transfer.PersonalInfo;
import gov.va.med.mhv.service.MHVAbstractService;
import gov.va.med.mhv.usermgmt.bizobj.BusinessObjectFactory;
import gov.va.med.mhv.usermgmt.bizobj.InPersonAuthenticationAssembler;
import gov.va.med.mhv.usermgmt.bizobj.InPersonAuthenticationBO;
import gov.va.med.mhv.usermgmt.bizobj.PatientAssembler;
import gov.va.med.mhv.usermgmt.bizobj.PatientBO;
import gov.va.med.mhv.usermgmt.enumeration.ActivityActorTypeEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.AuthenticationStatus;
import gov.va.med.mhv.usermgmt.enumeration.PatientCorrelationStatus;
import gov.va.med.mhv.usermgmt.integration.adapter.PatientCorrelationServiceAdapter;
import gov.va.med.mhv.usermgmt.messages.UserManagementMessages;
import gov.va.med.mhv.usermgmt.service.EntityMaintenanceService;
import gov.va.med.mhv.usermgmt.service.InPersonAuthenticationService;
import gov.va.med.mhv.usermgmt.service.InPersonAuthenticationServiceResponse;
import gov.va.med.mhv.usermgmt.service.PatientCollectionServiceResponse;
import gov.va.med.mhv.usermgmt.service.PatientService;
import gov.va.med.mhv.usermgmt.service.PatientServiceResponse;
import gov.va.med.mhv.usermgmt.service.ServiceFactory;
import gov.va.med.mhv.usermgmt.service.UserProfileServiceResponse;
import gov.va.med.mhv.usermgmt.service.adapter.AdapterException;
import gov.va.med.mhv.usermgmt.service.adapter.DuplicatePatientException;
import gov.va.med.mhv.usermgmt.service.adapter.LookupPatientAdapter;
import gov.va.med.mhv.usermgmt.service.adapter.LookupPatientAdapterWithMiddleName;
import gov.va.med.mhv.usermgmt.service.adapter.MpiProperties;
import gov.va.med.mhv.usermgmt.service.delegate.MviIntegrationServiceDelegate;
import gov.va.med.mhv.usermgmt.service.delegate.ServiceDelegateFactory;
import gov.va.med.mhv.usermgmt.service.handler.MviProperties;
import gov.va.med.mhv.usermgmt.transfer.Facility;
import gov.va.med.mhv.usermgmt.transfer.InPersonAuthentication;
import gov.va.med.mhv.usermgmt.transfer.Patient;
import gov.va.med.mhv.usermgmt.transfer.PatientPK;
import gov.va.med.mhv.usermgmt.transfer.PatientRegistryChange;
import gov.va.med.mhv.usermgmt.transfer.TransferObjectFactory;
import gov.va.med.mhv.usermgmt.transfer.UserProfile;
import gov.va.med.mhv.usermgmt.transfer.UserProfileDeactivationReason;
import gov.va.med.mhv.usermgmt.util.Auditor;
import gov.va.med.mhv.usermgmt.util.InPersonAuthenticationStatusUtils;
import gov.va.med.mhv.usermgmt.util.MessageKeys;
import gov.va.med.mhv.usermgmt.util.PHRAccessControl;
import gov.va.med.mhv.usermgmt.util.PatientCorrelationStatusUtils;
import gov.va.med.mhv.usermgmt.util.RegistryDescriptionBuilder;
import gov.va.med.mhv.usermgmt.util.UserProfileDeactivationUtils;

/**
 * Service implementation class for the Patient service
 */
public class PatientServiceImpl extends MHVAbstractService implements
	PatientService
{

	private static final Log LOG = LogFactory.getLog(PatientServiceImpl.class);

    private static String INVALIDATED_ICN_DELIMITER = "A";


    private PatientCorrelationServiceAdapter adapter =
        new PatientCorrelationServiceAdapter();

    @Override
    protected Log getLog() {
        return LOG;
    }

    /*
     * (non-Javadoc)
     * @see gov.va.med.mhv.usermgmt.service.PatientService#getPatientForUser(
     * gov.va.med.mhv.usermgmt.transfer.UserProfile)
     */
    public PatientServiceResponse getPatientForUser(UserProfile userProfile) {
        PatientServiceResponse response = new PatientServiceResponse();

        if ((userProfile != null)) {
            response.setPatient(getPatientForUserProfileId(userProfile.
                getId()));
        }

        return response;
    }

    /*
     * (non-Javadoc)
     * @see gov.va.med.mhv.usermgmt.service.PatientService#getPatientByIcn(
     * java.lang.String)
     */
    public PatientServiceResponse getPatientByIcn(String icn) {
        Precondition.assertNotBlank("icn", icn);
        PatientServiceResponse response = new PatientServiceResponse();

        List patientBOs = PatientBO.getPatientByIcn(icn);
        Collection patients = PatientAssembler.getPatientCollection(patientBOs);
        if (patients.size() > 0) {
            response.setPatient((Patient) patients.iterator().next());
        }

        return response;
    }

    /*
     * (non-Javadoc)
     * @see gov.va.med.mhv.usermgmt.service.PatientService#getPatientById(
     * java.lang.String)
     */
    public PatientServiceResponse getPatientById(Long id) {
        PatientServiceResponse response = new PatientServiceResponse();

        PatientBO patientBO = PatientBO.findByPrimaryKey(new PatientPK(id));
        Patient patient = patientBO.getPatientValues();
        response.setPatient(patient);

        return response;
    }

    /*
     * (non-Javadoc)
     * @see gov.va.med.mhv.usermgmt.service.PatientService
     * #createPatientForUser(gov.va.med.mhv.usermgmt.transfer.Patient,
     * gov.va.med.mhv.usermgmt.transfer.UserProfile)
     */
    public PatientServiceResponse createPatientForUser(Patient patient,
        UserProfile userProfile)
    {
        PatientServiceResponse response = new PatientServiceResponse();

        patient.setUserProfile(userProfile);
        PatientServiceResponse psr = ServiceFactory.
            createEntityMaintenanceService().save(patient);

        response.setPatient(psr.getPatient());
        response.getMessages().addMessages(psr.getMessages());

        return response;
    }

	/**
	 * Execute the AddFacilityForPatient service
	 * @return AddFacilityForPatientServiceResponse
	 */
	public PatientServiceResponse addFacilityForPatient(Patient patient,
		Facility facility)
	{
		PatientServiceResponse response = new PatientServiceResponse();

		PatientPK patientKey = (PatientPK) patient.getKey();
		EntityMaintenanceService ems = ServiceFactory
			.createEntityMaintenanceService();
		Patient p = ems.findByPrimaryKey(patientKey).getPatient();
		p.addFacility(facility);
		PatientServiceResponse patientServiceResponse = ems.save(p);

		response.setPatient(patientServiceResponse.getPatient());
        copyMessages(response, patientServiceResponse);

		return response;
	}

	/**
	 * This operation gets a patient and facilities from the patient registry
	 * system (MPI).
	 * @param firstName
	 * @param lastName
	 * @param birthDate
	 * @param ssn
	 * @return GetPatientFromRegistryServiceResponse contains the patient
	 */
	public PatientServiceResponse getPatientFromRegistry(String firstName,
		String lastName, Date birthDate, String ssn, Long userId)
	{
		PatientServiceResponse response = new PatientServiceResponse();

		String numberOnlySsn = StringUtils.replace(ssn, "-", "");
		LookupPatientAdapter adapter = new LookupPatientAdapter(firstName,
			lastName, birthDate, numberOnlySsn, userId);
		try {
			Patient patient = adapter.lookupPatient();
			response.setPatient(patient);
			if (patient == null) {
				LOG.error("No patient found in MPI for first name: "
					+ firstName + " last name: " + lastName + " birth date: "
					+ formatDate(birthDate) + " ssn: " + ssn);
			} else if (LOG.isDebugEnabled()) {
				LOG.info("Patient successfully matched with MPI for first name: "
					+ firstName + " last name: " + lastName + " birth date: "
					+ formatDate(birthDate) + " last four digits of ssn: "
					+ StringUtils.substring(numberOnlySsn, 5));
			}
		} catch (DuplicatePatientException e) {
			LOG.error("More than one patient found in MPI for first name: "
				+ firstName + " last name: " + lastName + " birth date: "
				+ formatDate(birthDate) + " last four digits of ssn: "
				+ StringUtils.substring(numberOnlySsn, 5) + "; "
				+ e.getLocalizedMessage());
			addError(response, MessageKeys.PATIENT_DUPLICATE_ENTRIES);
		} catch (AdapterException e) {
			LOG.error("MPI connection or timeout exception: Host="
				+ e.getHost() + ":" + String.valueOf(e.getPort()));
			addError(response,
				UserManagementMessages.PATIENT_REGISTRY_ACCESS_ERROR);
		}

		return response;
	}

	/**
	 * This operation gets a patient and facilities from the patient registry
	 * system (MPI).
	 * @param firstName
	 * @param lastName
	 * @param middleName
	 * @param gender
	 * @param birthDate
	 * @param ssn
	 * @return GetPatientFromRegistryWithMiddleNameServiceResponse contains the patient
	 */
	public PatientServiceResponse getPatientFromRegistryWithMiddleName(String firstName,
		String lastName, String middleName, String gender, Date birthDate, String ssn, Long userId)
	{
		PatientServiceResponse response = new PatientServiceResponse();

		String numberOnlySsn = StringUtils.replace(ssn, "-", "");
		LookupPatientAdapterWithMiddleName adapter = new LookupPatientAdapterWithMiddleName(firstName,
			lastName, middleName, gender, birthDate, numberOnlySsn, userId);
		try {
			Patient patient = adapter.lookupPatient();
			response.setPatient(patient);
			if (patient == null) {
				LOG.error("No patient found in MPI for FIRST name: "
					+ firstName + " LAST name: " + lastName + " MIDDLE name: " + middleName +" birth date: "
					+ formatDate(birthDate) + " GENDER: " + gender  + " last four digits of ssn: "
					+ StringUtils.substring(numberOnlySsn, 5));
			} else if (LOG.isInfoEnabled()) {
				LOG.info("Patient successfully matched with MPI for first name: "
					+ firstName + " last name: " + lastName + " middle name: " + middleName + " birth date: "
					+ formatDate(birthDate) + " gender: " + gender + " last four digits of ssn: "
					+ StringUtils.substring(numberOnlySsn, 5));
			}
		} catch (DuplicatePatientException e) {
			LOG.error("More than one patient found in MPI for first name: "
				+ firstName + " last name: " + lastName + " middle name: " + middleName + " birth date: "
				+ formatDate(birthDate) + " gender: " + gender + " last four digits of ssn: "
				+ StringUtils.substring(numberOnlySsn, 5) + "; "
				+ e.getLocalizedMessage());
			addError(response, MessageKeys.PATIENT_DUPLICATE_ENTRIES);
		} catch (AdapterException e) {
			LOG.error("MPI connection or timeout exception: Host="
				+ e.getHost() + ":" + String.valueOf(e.getPort()));
			addError(response,
				UserManagementMessages.PATIENT_REGISTRY_ACCESS_ERROR);
		}

		return response;
	}

	public PatientServiceResponse getPatientFromMVI(UserProfile userProfile, Boolean atRegistration) {
		return ServiceDelegateFactory.createMviIntegrationServiceDelegate().searchPersonInMVI(userProfile, atRegistration);
	}
	
    /*
     *  (non-Javadoc)
     * @see gov.va.med.mhv.usermgmt.service.PatientService#
     * updatePatientFacilities(java.lang.String)
     */
	public PatientServiceResponse updatePatientFacilities(String userName) {
		PatientServiceResponse response = new PatientServiceResponse();

        try {
    		UserProfile userProfile = ServiceFactory.createUserProfileService().
                getProfileForUser(userName).getUserProfile();
    		if (userProfile == null) {
    			addError(response, UserManagementMessages.USER_DOES_NOT_EXIST,
                    new String[] { userName });
    			return response;
    		}
            if (!BooleanUtils.isTrue(userProfile.getIsPatient())) {
    			return response;
    		}
            response = getPatientForUser(userProfile);
            if (hasErrorMessages(response)) {
                return response;
            }
            Patient patient = response.getPatient();
            if (patient == null) {
                return response;
            }
            BooleanServiceResponse accessResponse = PHRAccessControl.
                hasPhrAccess(userName);
            if (hasErrorMessages(accessResponse)) {
                copyMessages(response, accessResponse);
                return response;
            }
            boolean isLinkEnabled = MpiProperties.getInstance().isLinkEnabled()
                && BooleanUtils.isTrue(accessResponse.getBoolean());
            // If not linking to MPI, then we must keep refreshing
            // the registry information upon login

            if (isLinkEnabled && !PatientCorrelationStatusUtils.isUncorrelated(
                patient))
            {
                if (LOG.isInfoEnabled()) {
                    LOG.info("Not updating facilities of patient '" + userName
                        + "', because patient is not uncorrelated (status=" +
                        patient.getCorrelationStatus().getName() + ")");
                }
            } else if (needsInvalidation(patient)) {
                if (LOG.isInfoEnabled()) {
                    LOG.info("Invalidating patient '" + userName
                        + "', because patient has invalid ICN (icn=" +
                        patient.getIcn()
                        + ") and has not been invalidated yet");
                }
                invalidatePatient(patient, null, UserProfileDeactivationUtils.
                    MULTIPLE_ACCOUNTS_RESOLUTION);
            } else if (isLinkEnabled && isPendingSynchronization(patient)) {
                if (LOG.isInfoEnabled()) {
                    LOG.info("Correlate patient '" + userName +
                        ", because patient is pending synchronization");
                }
                // NOTE correlate also calls updatePatientRegistryInformation()
                correlate(patient);
            } else {
                if (LOG.isInfoEnabled()) {
                    LOG.info("Update patient facilities for '" + userName +
                        "' in updatePatientRegistryInformation()");
                }
                //Correlated patients do have PatientRegistry Information invoked during login
                updatePatientRegistryInformation(userProfile);
            }
            return response;
        } finally {
            // Logging messages here, because this operation is intended to
            // be called asynchronously, such that messages could get lost
            if (response.getMessages().hasErrorMessages()) {
                logMessages(response);
            }
        }
	}

    /* (non-Javadoc)
     * @see gov.va.med.mhv.usermgmt.service.PatientService#updatePatient(
     * java.lang.String)
     */
    public PatientServiceResponse updatePatientRegistryInformation(
        Patient patient)
    {
        if (LOG.isDebugEnabled()) {
            LOG.debug("updatePatientRegistryInformation(" + patient.
                getUserProfile().getUserName() + ")");
        }

        PatientServiceResponse response = new PatientServiceResponse();
        Patient patientToUpdate = getPatientForUserProfileId(patient.
            getUserProfile().getId());
        if (patientToUpdate == null) {
            addError(response, MessageKeys.PATIENT_NOT_FOUND);
        } else {
            // The patientToUpdate is the original patient that is updated
            // with the new information to avoid loss of data and
            // avoid updates to non-registry information.
            RegistryChangeProcessor processor =
                new RegistryChangeProcessor(patientToUpdate, patient);
            if (processor.hasAddedRegistryChange()) {
                if (!hasErrorMessages(response)) {
                    response = savePatient(patientToUpdate, processor.
                        hasDemographicInfoChange());
                }
            } else {
                response.setPatient(patientToUpdate);
            }
        }

        return response;
    }

	//MHV_CodeCR1514 - US12.4 MVI Compliance Implementation - Adding new method calls for MVI 
    public BooleanServiceResponse mviAuthenticate(Patient patient) {
        return	ServiceFactory.createMviIntegrationService().mviAuthenticate(patient, patient.getUserProfile(), true, false);
    }
    
    /*
     *  (non-Javadoc)
     * @see gov.va.med.mhv.usermgmt.service.PatientService#correlate(
     * gov.va.med.mhv.usermgmt.transfer.Patient)
     */
    public VoidServiceResponse correlate(Patient patient) {
        if (!existsPatient(patient)) {
            return createPatientNotFoundResponse();
        }
        BooleanServiceResponse accessResponse = PHRAccessControl.hasPhrAccess(
            patient);
        if (hasErrorMessages(accessResponse)) {
            return createVoidServiceResponseFrom(accessResponse);
        }
        boolean hasPhrAccess = accessResponse.getBoolean();
        boolean isLinkEnabled = MpiProperties.getInstance().isLinkEnabled();

        if (!(isLinkEnabled  && hasPhrAccess)) {
            if (LOG.isDebugEnabled()) {
                String reason = (!isLinkEnabled)
                    ? "correlation is disabled (mpi.properties)"
                    : "patient access to PHR is disabled (access controls)";
                LOG.debug("Not correlating patient " +
                    RegistryDescriptionBuilder.describe(patient) +
                    ", because " + reason +
                    ". Mark for synchronization instead.");
            }
            //patient have phr access means they are authenticated so I guess they are 
            //			getting added in patient_synchronization table. 
            patient.addPatientSynchronization(TransferObjectFactory.
                createPatientSynchronization());
            VoidServiceResponse response = new VoidServiceResponse();
            copyMessages(response, savePatient(patient, false));
            return response;
        }

        if (!PatientCorrelationStatusUtils.isUncorrelated(patient)
            // Allow retries
            && !PatientCorrelationStatusUtils.isFailedCorrelation(patient))
        {
            return createPatientNotUncorrelatedResponse();
        }
        // Synchronize information with MPI (via the query adapter),
        // such that the patient registration is up-to-date
        // and we do not get matching failures upon correlation
        // TODO: Refactor so this can be called directly and the patient
        // gets saved only in this method
        PatientServiceResponse queryResponse = updatePatientRegistryInformation(
            patient.getUserProfile());
        if (hasErrorMessages(queryResponse)) {
            return createVoidServiceResponseFrom(queryResponse);
        }
        if (queryResponse.getPatient() == null) {
            return createPatientRegistryAccessResponse();
        }
        patient = queryResponse.getPatient();

        patient.setCorrelationStatus(PatientCorrelationStatusUtils.
            PENDING_CORRELATION);
        PatientServiceResponse response = savePatient(patient, true);

        if (hasErrorMessages(response)) {
            return createVoidServiceResponseFrom(response);
        }

        if (LOG.isDebugEnabled()) {
            LOG.debug("Request correlation with MPI for patient "
                + RegistryDescriptionBuilder.describe(patient));
        }
        return adapter.correlate(patient);
    }

    /*
     * (non-Javadoc)
     * @see gov.va.med.mhv.usermgmt.service.PatientService#finalizeCorrelation(
     * gov.va.med.mhv.usermgmt.transfer.Patient, java.lang.String)
     */
    public VoidServiceResponse finalizeCorrelation(Patient patient,
        String error)
    {
        if (!existsPatient(patient)) {
            return createPatientNotFoundResponse();
        }
        if (!PatientCorrelationStatusUtils.isPendingCorrelation(patient)) {
            return createPatientNotPendingCorrelationResponse();
        }
        VoidServiceResponse response = new VoidServiceResponse();
        boolean hasError = !StringUtils.isBlank(error);
        if (hasError) {
            patient.setCorrelationStatus(PatientCorrelationStatusUtils.
                FAILED_CORRELATION);
            patient.setCorrelationErrorCode(error);
            // Do not add an 'error' message, as it will rollback the
            // transaction and the error must be captured in the database
            addInfo(response, MessageKeys.PATIENT_MPI_CORRELATION_FAILED,
                new Object[] { patient.getIcn(), error });
        } else {
            patient.setCorrelationErrorCode(null);
            patient.setCorrelationStatus(PatientCorrelationStatusUtils.
                CORRELATED);
            // Remove any patient synchronizations, avoiding further
            // synchronization attempts upon login
            // NOTE This code could be removed once all prior authenticated
            // patients are synchronized (i.e. correlated)
            patient.setPatientSynchronizations(null);
        }
        patient.getUserProfile().setIsMPIControlled(true);
        PatientServiceResponse patientServiceResponse = savePatient(patient,
            true);
        if (hasErrorMessages(patientServiceResponse)) {
            addError(response, MessageKeys.PATIENT_CORRELATION_UPDATE_FAILED,
                new Object[] { patient.getIcn()});
        } else if (!hasError) {
            addInfo(response, MessageKeys.PATIENT_MPI_CORRELATION_SUCCEEDED,
                new Object[] { patient.getIcn(), "finalize correlation" });
        }
        copyMessages(response, patientServiceResponse);
        return response;
    }

    /*
     * (non-Javadoc)
     * @see gov.va.med.mhv.usermgmt.service.PatientService#uncorrelate(
     * gov.va.med.mhv.usermgmt.transfer.Patient)
     */
    public VoidServiceResponse uncorrelate(Patient patient) {
        if (!existsPatient(patient)) {
            return createPatientNotFoundResponse();
        }
        boolean isEnabled = MpiProperties.getInstance().isUnlinkEnabled();
        boolean canUncorrelate =
            PatientCorrelationStatusUtils.isCorrelated(patient)
            // Allow retries
            || PatientCorrelationStatusUtils.isFailedUncorrelation(patient);
        boolean mustUnlink = isEnabled && canUncorrelate;

        patient.setCorrelationStatus((mustUnlink)
            ? PatientCorrelationStatusUtils.PENDING_UNCORRELATION
            : PatientCorrelationStatusUtils.UNCORRELATED);
        patient.setPatientSynchronizations(null);
        PatientServiceResponse response = savePatient(patient, true);
        if (hasErrorMessages(response) || !mustUnlink) {
            return createVoidServiceResponseFrom(response);
        }
        if (LOG.isInfoEnabled()) {
            LOG.info("Request uncorrelation from MPI for patient " +
                RegistryDescriptionBuilder.describe(patient));
        }
        return adapter.uncorrelate(patient);
    }

	//MHV_CodeCR1514 - US12.4 MVI Compliance Implementation - Adding new method calls for MVI 
    public BooleanServiceResponse mviUnauthenticate(Patient patient) {
        return	ServiceFactory.createMviIntegrationService().mviAuthenticate(patient, patient.getUserProfile(), false, false);
    }
    
    /*
     * (non-Javadoc)
     * @see gov.va.med.mhv.usermgmt.service.PatientService#
     * finalizeUncorrelation(gov.va.med.mhv.usermgmt.transfer.Patient,
     * java.lang.String)
     */
    public VoidServiceResponse finalizeUncorrelation(Patient patient,
        String error)
    {
        if (!existsPatient(patient)) {
            return createPatientNotFoundResponse();
        }
        if (!PatientCorrelationStatusUtils.isPendingUncorrelation(patient)) {
            return createPatientNotPendingUncorrelationResponse();
        }
        VoidServiceResponse response = new VoidServiceResponse();
        boolean hasError = !StringUtils.isBlank(error);
        if (hasError) {
            patient.setCorrelationStatus(PatientCorrelationStatusUtils.
                FAILED_UNCORRELATION);
            patient.setCorrelationErrorCode(error);
            // Do not use an error message, as it will rollback the
            // transaction and the error must be captured in the database
            addInfo(response, MessageKeys.PATIENT_MPI_UNCORRELATION_FAILED,
                new Object[] { patient.getIcn(), error });
        } else {
            patient.setCorrelationStatus(PatientCorrelationStatusUtils.
                UNCORRELATED);
        }
        // NOTE: We cannot set isMPICorrelated back to false, as opposed to
        // setting it to true in the finalizeCorrelation, because it turns the
        // validation back and the MPI controlled values may no longer be valid
        // (the reason to use the isMPIControlled flag)
        // The flag needs to be set to false, once the user modifies the
        // profile and saves it
        PatientServiceResponse patientServiceResponse = savePatient(patient,
            false);
        copyMessages(response, patientServiceResponse);
        if (hasErrorMessages(patientServiceResponse)) {
            addError(response, MessageKeys.PATIENT_CORRELATION_UPDATE_FAILED,
                new Object[] { patient.getIcn(), "finalize uncorrelation"});
        } else {
            addInfo(response, MessageKeys.PATIENT_MPI_UNCORRELATION_SUCCEEDED,
                new Object[] { patient.getIcn() });
        }
        return response;
    }

    /*
     *  (non-Javadoc)
     * @see gov.va.med.mhv.usermgmt.service.PatientService#removeCorrelation(
     * gov.va.med.mhv.usermgmt.transfer.Patient)
     */
    public VoidServiceResponse removeCorrelation(Patient patient) {
        if (!existsPatient(patient)) {
            return createPatientNotFoundResponse();
        }
        VoidServiceResponse response = new VoidServiceResponse();

        patient.setCorrelationStatus(PatientCorrelationStatusUtils.
            UNCORRELATED);
        PatientServiceResponse patientServiceResponse = savePatient(
            patient, false);
        copyMessages(response, patientServiceResponse);
        if (hasErrorMessages(patientServiceResponse)) {
            addError(response, MessageKeys.PATIENT_CORRELATION_UPDATE_FAILED,
                new Object[] { patient.getIcn(), "remove correlation"});
        } else {
            addInfo(response, MessageKeys.PATIENT_MPI_REMOVAL_SUCCEEDED,
                new Object[] { patient.getIcn() });
        }
        return response;
    }
    
    public PatientServiceResponse invalidatePatient(Patient patient,
        String referenceId, UserProfileDeactivationReason reason)
    {
        PatientServiceResponse response = new PatientServiceResponse();
        Precondition.assertNotNull("patient", patient);
        Precondition.assertNotBlank("patient.icn", patient.getIcn());

		PatientServiceResponse mviSyncServiceResponse = 
			ServiceFactory.createMviIntegrationService().deletePersonFromCorrelationForHelpDesk(
					patient, patient.getUserProfile(), "");
        if (hasErrorMessages(mviSyncServiceResponse)) {
            logMessages(mviSyncServiceResponse);
            LOG.error("Unable to deactivate account of DFN ID: " +
                    patient.getUserProfile().getId() + 
                    " because MVI uncorrelate was not successful ");
            return response;
        }
        
        String invalidatedICN = null;

        if (!hasInvalidatedIcn(patient)) {
            // Only invalidate, if ICN Not yet invalidated
            invalidatedICN = patient.getIcn();
            patient.setIcn(createInvalidIcn(patient.getIcn()));
            response = updatePatientRegistryInformation(patient);
            patient = response.getPatient();
        }
        if (hasErrorMessages(response)) {
            logMessages(response);
            return response;
        }
        
        // Update all non-registry patient information
        patient.setFacilitys(null);
        patient.setCorrelationStatus(PatientCorrelationStatusUtils.CORRELATION_INVALID);
        if (!StringUtils.isBlank(invalidatedICN)) {
            patient.setInvalidatedIcn(invalidatedICN);
        }
        patient.setInvalidationReferenceId(StringUtils.isBlank(referenceId)
            ? createReferenceId(patient) : referenceId);
        patient.setPatientSynchronizations(null);
        response = savePatient(patient, false);
        patient = response.getPatient();

        if (hasErrorMessages(response)) {
            logMessages(response);
        } else if (reason == null) {
            // We do not want to fail this action
            LOG.error("Unable to deactivate account of patient " +
                patient.getUserProfile().getUserName() + ", because the " +
                " deactivation reason instance is unknown (referenceId: " +
                referenceId + ")");
             // TODO Add info message
        } else {
            if (LOG.isInfoEnabled()) {
                LOG.info("Invalidated patient '" + patient.getUserProfile().
                    getUserName() + "'. Changed ICN from " + patient.
                    getInvalidatedIcn() + " to " + patient.getIcn());
            }
            // We do not want account deactivation failure to prevent
            // this transaction from succeeding.
            // The PHR access has already been disabled, by changing the ICN
            logMessages(ServiceFactory.createUserProfileService().
                changeActivationState(patient.getUserProfile(), reason,
                UserProfileDeactivationUtils.STATE_CHANGED_BY_SYSTEM, reason.getName(), null));
        }
        return response;
    }
    
    //Admin Portal_CodeCR1959 - React/Deact Impl
	public PatientServiceResponse reactivatePatientInHelpDesk(Patient patient,
        String adminNetworkId, UserProfileDeactivationReason reason, String note) {
		
		PatientServiceResponse response = new PatientServiceResponse();
		if(patient.getInvalidatedIcn() == null) {
            addError(response, "mvi.helpdesk.revalidate.error.patient.active", new Object[] {});
		}
		
        if (reason == null) {
	   		addError(response, "mvi.helpdesk.revalidate.error.reason.missing", new String[]{"reactivateReason"});
	        if (LOG.isDebugEnabled()) {
	           LOG.debug("No deactivation reason for user:" + patient.getUserProfile().getUserName());
	
	        }
	    }
	       
	    if ((note == null)||(note.length()==0)) {
	    	addError(response, "mvi.helpdesk.revalidate.error.description.missing", new String[]{"reactivateNote"});
	    }
        
        if (hasErrorMessages(response)) {
            logMessages(response);
            return response;
        }
        
		patient.setIcn(patient.getInvalidatedIcn().trim());
		patient.setInvalidatedIcn(null);
		patient.setInvalidationReferenceId(createReferenceId(patient));
        response = savePatient(patient, false);
	
        VoidServiceResponse vResponse = ServiceFactory.createUserProfileService().
        reactivateUserInHelpDesk(patient.getUserProfile(), reason, adminNetworkId, note);
        if (!hasErrorMessages(vResponse)) {
            logMessages(vResponse);
        } else {
	    	addError(response, "mvi.helpdesk.revalidate.fail", new String[]{getFullName(patient.getUserProfile())});
        }
    	return response;
    }
    
	private String getFullName(UserProfile userProfile) {
		String firstName = userProfile.getFirstName();
		String middleName = userProfile.getMiddleName();
		String lastName = userProfile.getLastName();
		
		StringBuffer buff = new StringBuffer();
		buff.append(firstName);
		if (middleName != null) {
			buff.append(' ');
			buff.append(middleName);
		}
		buff.append(' ');
		buff.append(lastName);
		
		return(buff.toString());
	}

    private boolean isFourOrMoreTraitsUpdate(gov.va.med.mhv.usermgmt.transfer.Patient patient, PersonalInfo personalInfo) {
    	int traitUpdateCount = 0;
    	
        UserProfile userProfile = patient.getUserProfile();
        
        if (LOG.isDebugEnabled()) {
            LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() for user profile id:" + userProfile.getId());
        }
        
        if(!userProfile.getFirstName().equalsIgnoreCase(personalInfo.getName().getFirstName())) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() first name is different for user profile id:" + userProfile.getId());
            }
        	traitUpdateCount++;
        }
        
        if(userProfile.getMiddleName()==null && personalInfo.getName().getMiddleName()!=null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() middle name is different for user profile id:" + userProfile.getId());
            }
        	traitUpdateCount++;
        } else if(userProfile.getMiddleName()!=null && personalInfo.getName().getMiddleName()==null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() middle name is different for user profile id:" + userProfile.getId());
            }
        	traitUpdateCount++;
        } else if(userProfile.getMiddleName()==null && personalInfo.getName().getMiddleName()==null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() no middle name for user profile id:" + userProfile.getId());
            }
        } else if(!userProfile.getMiddleName().equalsIgnoreCase(personalInfo.getName().getMiddleName())) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() middle name is different for user profile id:" + userProfile.getId());
            }
        	traitUpdateCount++;
        }
        
        if(!userProfile.getLastName().equalsIgnoreCase(personalInfo.getName().getLastName())) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() last name is different for user profile id:" + userProfile.getId());
            }
        	traitUpdateCount++;
        }
        
        if(!userProfile.getBirthDate().equals(personalInfo.getDateOfBirth())) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() birth date is different for user profile id:" + userProfile.getId());
            }
        	traitUpdateCount++;
        }
        
        if(!userProfile.getGender().toString().equalsIgnoreCase(personalInfo.getGender().toString())) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() gender is different for user profile id:" + userProfile.getId());
            }
        	traitUpdateCount++;
        }

        if(!userProfile.getSsn().equals(personalInfo.getSsn())) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() ssn is different for user profile id:" + userProfile.getId());
            }
        	traitUpdateCount++;
        }
        
        if(traitUpdateCount>3) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("@@@@@@@@@@@@@@@@@@@@@@@@@@@@ traitUpdateCount>3");
            }
        	return true;
        }
        
        return false;
    }
	
    //JAZZ Defect# 20309: If there four or more traits update, DO NOT Update the user and deactivate the user.
	public PatientServiceResponse deactivatePatientForMultiTraitUpdateError(Patient patient,
            String referenceId, String adminNetowrkId, UserProfileDeactivationReason reason, String note) {
        Precondition.assertNotNull("patient", patient);
        Precondition.assertNotBlank("patient.icn", patient.getIcn());

        PatientServiceResponse response = new PatientServiceResponse();
        
        if (reason == null) {
	   		addError(response, "mvi.multi.trait.update.error.invalidate.error.reason.missing", new String [] {"deactivateReason"});
	        if (LOG.isDebugEnabled()) {
	           LOG.debug("No deactivation reason for user:" + patient.getUserProfile().getUserName());
	
	        }
	    }
	       
	    if ((note == null)||(note.length()==0)) {
	    	addError(response, "mvi.multi.trait.update.error.invalidate.error.reason.missing", new String [] {"deactivateNote"});
	    }
        
        if (hasErrorMessages(response)) {
            logMessages(response);
            return response;
        }
        
        String invalidatedICN = null;

        if (!hasInvalidatedIcn(patient)) {
            // Only invalidate, if ICN Not yet invalidated
            invalidatedICN = patient.getIcn();
            patient.setIcn(createInvalidIcn(patient.getIcn()));
            response = updatePatientRegistryInformation(patient);
            patient = response.getPatient();
        }
        
        //JAZZ: 19292 - Take out matched date, set mpi controlled flag to null during deactivation.
        patient.setMatchedDateTime(null);
        
        if (hasErrorMessages(response)) {
            logMessages(response);
            return response;
        }

        // Update all non-registry patient information
        patient.setFacilitys(null);
        if (!StringUtils.isBlank(invalidatedICN)) {
            patient.setInvalidatedIcn(invalidatedICN);
        }
        patient.setInvalidationReferenceId(StringUtils.isBlank(referenceId)
            ? createReferenceId(patient) : referenceId);
        patient.setPatientSynchronizations(null);
        response = savePatient(patient, false);
        patient = response.getPatient();

        if (hasErrorMessages(response)) {
            logMessages(response);
        } else if (reason == null) {
            // We do not want to fail this action
            LOG.error("Unable to deactivate account of patient " +
                patient.getUserProfile().getUserName() + ", because the " +
                " deactivation reason instance is unknown (referenceId: " +
                referenceId + ")");
             // TODO Add info message
        } else {
            if (LOG.isInfoEnabled()) {
                LOG.info("Invalidated patient '" + patient.getUserProfile().
                    getUserName() + "'. Changed ICN from " + patient.
                    getInvalidatedIcn() + " to " + patient.getIcn());
            }
            // We do not want account deactivation failure to prevent
            // this transaction from succeeding.
            // The PHR access has already been disabled, by changing the ICN
            VoidServiceResponse vResponse = ServiceFactory.createUserProfileService().
                changeActivationState(patient.getUserProfile(), reason, adminNetowrkId, note, null);
            if (hasErrorMessages(vResponse)) {
    	    	addError(response, "mvi.multi.trait.update.error.invalidate.fail", new String[]{getFullName(patient.getUserProfile())});
            } else {
    	    	logMessages(vResponse);
            }
        }
        return response;
		
	}
	
    private InPersonAuthenticationService getIPAService() {
        return ServiceFactory.createInPersonAuthenticationService();
    }   
    
    private void handleMessages(String description, EntityServiceResponse response) {
//        if (response.getMessages().hasInformationalMessages()) {
        	StringBuffer buffer = new StringBuffer();
        	buffer.append(new MessagesStringBuilder().append(response.getMessages(), getClass()).getInfoString());
        	throw new RuntimeException(description + buffer.toString());
        	//This is an old code that didn't have any description. Left it commented for research purpse            	
        	//throw new RuntimeException(MessageKeys.UNKNOWN_EXCEPTION_OCCURRED);
//        }
    }

    public PatientServiceResponse deactivatePatientInHelpDesk(Patient patient,
            String referenceId, String adminNetowrkId, UserProfileDeactivationReason reason, String note, Date deathDate) {
        Precondition.assertNotNull("patient", patient);
        Precondition.assertNotBlank("patient.icn", patient.getIcn());
        
        PatientServiceResponse response = new PatientServiceResponse();
        
        if (reason == null) {
	   		addError(response, "mvi.helpdesk.invalidate.error.reason.missing", new String [] {"deactivateReason"});
	        if (LOG.isDebugEnabled()) {
	           LOG.debug("No deactivation reason for user:" + patient.getUserProfile().getUserName());
	
	        }
	    }
	       
	    if ((note == null)||(note.length()==0)) {
	    	addError(response, "mvi.helpdesk.invalidate.error.description.missing", new String [] {"deactivateNote"});
	    }
        
        if (hasErrorMessages(response)) {
            logMessages(response);
            return response;
        }
        
        String invalidatedICN = null;

        if (!hasInvalidatedIcn(patient)) {
            // Only invalidate, if ICN Not yet invalidated
            invalidatedICN = patient.getIcn();
            patient.setIcn(createInvalidIcn(patient.getIcn()));
            response = updatePatientRegistryInformation(patient);
            patient = response.getPatient();
        }
        
        //JAZZ: 19292 - Take out matched date, set mpi controlled flag to null during deactivation.
        patient.setMatchedDateTime(null);
        
        if (hasErrorMessages(response)) {
            logMessages(response);
            return response;
        }

        // Update all non-registry patient information
        patient.setFacilitys(null);
        if (!StringUtils.isBlank(invalidatedICN)) {
            patient.setInvalidatedIcn(invalidatedICN);
        }
        patient.setInvalidationReferenceId(StringUtils.isBlank(referenceId)
            ? createReferenceId(patient) : referenceId);
        patient.setPatientSynchronizations(null);
        response = savePatient(patient, false);
        patient = response.getPatient();

        if (hasErrorMessages(response)) {
            logMessages(response);
        } else if (reason == null) {
            // We do not want to fail this action
            LOG.error("Unable to deactivate account of patient " +
                patient.getUserProfile().getUserName() + ", because the " +
                " deactivation reason instance is unknown (referenceId: " +
                referenceId + ")");
             // TODO Add info message
        } else {
            if (LOG.isInfoEnabled()) {
                LOG.info("Invalidated patient '" + patient.getUserProfile().
                    getUserName() + "'. Changed ICN from " + patient.
                    getInvalidatedIcn() + " to " + patient.getIcn());
            }
            // We do not want account deactivation failure to prevent
            // this transaction from succeeding.
            // The PHR access has already been disabled, by changing the ICN
            VoidServiceResponse vResponse = ServiceFactory.createUserProfileService().
                changeActivationState(patient.getUserProfile(), reason, adminNetowrkId, note, deathDate);
            if (hasErrorMessages(vResponse)) {
    	    	addError(response, "mvi.helpdesk.invalidate.fail", new String[]{getFullName(patient.getUserProfile())});
            } else {
    	    	logMessages(vResponse);
            }
        }
        return response;
    }

    private String createReferenceId(Patient patient) {
        final SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        return format.format(new Date()) + StringUtils.substring(patient.
            getUserProfile().getSsn(),5);
    }

    private boolean needsInvalidation(Patient patient) {
        return (hasInvalidatedIcn(patient)
            && !PatientCorrelationStatusUtils.isCorrelationInvalid(
                    patient));
    }

    private boolean hasInvalidatedIcn(Patient patient) {
        assert patient != null;
        assert !StringUtils.isBlank(patient.getIcn());
        return !StringUtils.isBlank(patient.getInvalidatedIcn())
            || (patient.getIcn().indexOf(INVALIDATED_ICN_DELIMITER) > 0);

    }

    private String createInvalidIcn(String icn) {
        assert !StringUtils.isBlank(icn);
        
        final String icnPrefix = StringUtils.substring(icn, 0, 10)
            + INVALIDATED_ICN_DELIMITER;
        for (int i = 1; i < 999999; i++) {
            String newIcn = icnPrefix + String.format("%1$06d", i);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Assign new icn '" + newIcn + "'");
            }
            List patients = PatientBO.getPatientByIcn(newIcn);
            if ((patients == null) || patients.isEmpty()) {
                return newIcn;
            }
        }
        Precondition.fail("Cannot invalidate icn '" + icn +
            "', because too many invalidated already");
        return null;
    }

    private PatientServiceResponse updatePatientRegistryInformation(
        UserProfile userProfile)
    {
        PatientServiceResponse response = new PatientServiceResponse();
        PatientServiceResponse patientRegistryResponse = null;

        //Patient does not exist yet or is uncorrelated

		/*
		 * Checking the property file to see if we need to call getPatientFromRegistryWithMiddleName or getPatientFromRegistry
		 */
        boolean isUseMiddleNameAndGender = MpiProperties.getInstance().isUseMiddleNameAndGender();
        if(isUseMiddleNameAndGender){
	        if (LOG.isDebugEnabled()) {
	            LOG.debug("Middle name: " + userProfile.getMiddleName()
	            		+ "gender: " + userProfile.getGender().getName()
	                + " for user id: " + userProfile.getId()
	                + " and useMiddleNameAndGender property value: " + isUseMiddleNameAndGender + " from mpi.properties in updatePatientRegistryInformation method");
	        }
			String gender  = userProfile.getGender().getName();
			if(gender.equalsIgnoreCase("Male")){
				gender = "M";
			}else{
				gender = "F";
			}

	        patientRegistryResponse =
	        	getPatientFromRegistryWithMiddleName(userProfile.getFirstName(),
	            userProfile.getLastName(), userProfile.getMiddleName(), gender, userProfile.getBirthDate(),
	            userProfile.getSsn(), userProfile.getId());
		}else{
	        if (LOG.isDebugEnabled()) {
	            LOG.debug("isUseMiddleNameAndGender: " + isUseMiddleNameAndGender + " in PatientServiceImpl::updatePatientRegistryInformation for user id: " + userProfile.getId() + " not calling middle name service");
	        }
	        patientRegistryResponse =
	            getPatientFromRegistry(userProfile.getFirstName(),
	            userProfile.getLastName(), userProfile.getBirthDate(),
	            userProfile.getSsn(), userProfile.getId());
		}



        if (hasErrorMessages(patientRegistryResponse)) {
            copyMessages(response, patientRegistryResponse);
            return response;
        }
        if (patientRegistryResponse.getPatient() == null) {
            addError(response, MessageKeys.PATIENT_NOT_FOUND);
            return response;
        }

        Patient patient = patientRegistryResponse.getPatient();
        patient.setUserProfile(userProfile);
        response = updatePatientRegistryInformation(patient);
        return response;
    }

    /**
     * Determine if a given patient is pending synchronization.
     * @param patient The patient to check
     * @return True, only if the given patient is not null, has a
     * PatientSynchronization associated with it and is still uncorrelated.
     */
    private boolean isPendingSynchronization(
        gov.va.med.mhv.usermgmt.transfer.Patient patient)
    {
        return (patient != null)
            && (patient.getPatientSynchronizations() != null)
            && !patient.getPatientSynchronizations().isEmpty()
            // Correlation could already be in progress
            && PatientCorrelationStatusUtils.isUncorrelated(patient);
    }

    private Patient getPatientForUserProfileId(Long id) {
        Patient patient = null;
        if (id != null) {
            List patientBOs = PatientBO.getPatientForUserProfile(id);
            Collection patients = PatientAssembler.getPatientCollection(
                patientBOs);
            if (patients.size() > 0) {
                patient = (Patient) patients.iterator().next();
            }
        }
        return patient;
    }

    private PatientServiceResponse savePatient(Patient patient,
        boolean saveUserProfile)
    {
        PatientServiceResponse patientServiceResponse = ServiceFactory.
            createEntityMaintenanceService().save(patient);
        if (!hasErrorMessages(patientServiceResponse) && saveUserProfile) {
            UserProfileServiceResponse userProfileServiceResponse =
                ServiceFactory.createEntityMaintenanceService().save(
                    patient.getUserProfile());
            copyMessages(patientServiceResponse, userProfileServiceResponse);
        }
        if (hasErrorMessages(patientServiceResponse)) {
            logMessages(patientServiceResponse);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("Saved patient " +
                ((saveUserProfile) ? "and user profile " : "") +
                "changes: " + RegistryDescriptionBuilder.describe(
                   patientServiceResponse.getPatient()));
        }
        return patientServiceResponse;
    }

    private VoidServiceResponse createVoidServiceResponseFrom(
        ServiceResponse copyFrom)
    {
        VoidServiceResponse response = new VoidServiceResponse();
        copyMessages(response, copyFrom);
        return response;
    }
    private VoidServiceResponse createVoidServiceResponseFrom(
        EntityServiceResponse copyFrom)
    {
        VoidServiceResponse response = new VoidServiceResponse();
        copyMessages(response, copyFrom);
        return response;
    }

    private VoidServiceResponse createPatientNotFoundResponse() {
        VoidServiceResponse response = new VoidServiceResponse();
        addError(response, MessageKeys.PATIENT_NOT_FOUND);
        return response;
    }

    private VoidServiceResponse createPatientRegistryAccessResponse() {
        VoidServiceResponse response = new VoidServiceResponse();
        addError(response, UserManagementMessages.PATIENT_REGISTRY_ACCESS_ERROR);
        return response;
    }

    private VoidServiceResponse createPatientNotUncorrelatedResponse() {
        VoidServiceResponse response = new VoidServiceResponse();
        addError(response, MessageKeys.PATIENT_IS_NOT_UNCORRELATED);
        return response;
    }

    private VoidServiceResponse createPatientNotPendingCorrelationResponse() {
        VoidServiceResponse response = new VoidServiceResponse();
        addError(response, MessageKeys.PATIENT_IS_NOT_PENDING_CORRELATION);
        return response;
    }

    private VoidServiceResponse createPatientNotPendingUncorrelationResponse() {
        VoidServiceResponse response = new VoidServiceResponse();
        addError(response, MessageKeys.PATIENT_IS_NOT_PENDING_UNCORRELATION);
        return response;
    }

    private boolean existsPatient(Patient patient) {
        assertPatientIds(patient);
        return PatientBO.findByPrimaryKey(patient.getPatientPK()) != null;
    }

    private void assertPatientIds(Patient patient) {
        Precondition.assertNotNull("patient", patient);
        Precondition.assertNotNull("patient.id", patient.getId());
        Precondition.assertNotNull("patient.userProfile", patient.
            getUserProfile());
        Precondition.assertNotNull("patient.userProfile.id", patient.
            getUserProfile().getId());
    }

    private String formatDate(Date d) {
		if(d==null) return "null";
		String bdate = d.toString();
		try {
		  SimpleDateFormat df = new SimpleDateFormat("yyyyMMdd");
	          bdate = df.format(d);
		} catch(Exception e) {}
		return bdate;
    }


    private final class RegistryChangeProcessor {

        private final Patient oldPatient;
        private final Patient newPatient;
        private final UserProfile newUserProfile;
        private final UserProfile oldUserProfile;
        private PatientRegistryChange changeRecord = null;
        private boolean hasDemographicChange = false;

        public RegistryChangeProcessor(Patient oldPatient, Patient newPatient) {
            this.oldPatient = oldPatient;
            this.newPatient = newPatient;
            this.oldUserProfile = oldPatient.getUserProfile();
            this.newUserProfile = newPatient.getUserProfile();
            processIcnChange();
            processNameChange();
            processBirthDateChange();
            processSsnChange();
            processGenderChange();
            // Not processing address changes until all information is sent
            // by MPI; Once implemented, remove suppress unused
            //processAddressChange();
            processContactInfoChange();
            processFacilityChanges();
            if (LOG.isDebugEnabled()){
                LOG.debug("hasChanges(" + oldUserProfile.getUserName() + ") = "
                    + hasAddedRegistryChange());
            }
            if (hasAddedRegistryChange()) {
                oldPatient.addPatientRegistryChange(changeRecord);
                LOG.debug("addedChange to " + DescriptionBuilder.describe(
                    oldPatient));
            }

        }

        public boolean hasAddedRegistryChange() {
            return changeRecord != null;
        }

        private void setDemographicInfoChange() {
            hasDemographicChange = true;
        }
        public boolean hasDemographicInfoChange() {
            return (changeRecord != null) && hasDemographicChange;
        }

        private PatientRegistryChange getChangeRecord(String forProperty) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("found " + forProperty + " change for "
                    + oldUserProfile.getUserName());
            }
            if (changeRecord == null) {
                changeRecord = TransferObjectFactory
                    .createPatientRegistryChange();
                changeRecord.setRecordedOnDate(TimestampUtils.
                    createCurrentTime());
            }
            return changeRecord;
        }

        private void processIcnChange() {
            if (!StringUtils.equals(newPatient.getIcn(), oldPatient.getIcn())) {
                getChangeRecord("ICN").setOldIcn(oldPatient.getIcn());
                oldPatient.setIcn(newPatient.getIcn());
            }
        }

        private void processNameChange() {
            // Remove suppress unused from method once activating
            // processing of middle name, prefix and suffix,
            processFirstNameChange();
            processMiddleNameChange();
            processLastNameChange();
            processNamePrefixChange();
            processNameSuffixChange();
        }
        private void processFirstNameChange() {
            if (!StringUtils.equals(newUserProfile.getFirstName(),
                oldUserProfile.getFirstName()))
            {
                getChangeRecord("FirstName").setOldFirstName(oldUserProfile.
                    getFirstName());
                oldUserProfile.setFirstName(newUserProfile.getFirstName());
                setDemographicInfoChange();
            }
        }

        private void processMiddleNameChange() {
            if (!StringUtils.equals(newUserProfile.getMiddleName(),
                oldUserProfile.getMiddleName()))
            {
                getChangeRecord("MiddleName").setOldMiddleName(oldUserProfile.
                    getMiddleName());
                oldUserProfile.setMiddleName(newUserProfile.getMiddleName());
                setDemographicInfoChange();
            }
        }

        private void processLastNameChange() {
            if (!StringUtils.equals(newUserProfile.getLastName(),
                oldUserProfile.getLastName()))
            {
                getChangeRecord("LastName").setOldLastName(oldUserProfile.
                    getLastName());
                oldUserProfile.setLastName(newUserProfile.getLastName());
                setDemographicInfoChange();
            }
        }

        private void processNamePrefixChange() {
            if (!equals(newUserProfile.getTitle(), oldUserProfile.getTitle())) {
                getChangeRecord("Title").setOldNamePrefix(oldUserProfile.
                    getTitle());
                oldUserProfile.setTitle(newUserProfile.getTitle());
                setDemographicInfoChange();
            }
        }
        private void processNameSuffixChange() {
            if (!equals(newUserProfile.getSuffix(), oldUserProfile.getSuffix())) {
                getChangeRecord("Suffix").setOldNameSuffix(oldUserProfile.
                    getSuffix());
                oldUserProfile.setSuffix(newUserProfile.getSuffix());
                setDemographicInfoChange();
            }
        }


        private void processBirthDateChange() {
            if (!newUserProfile.getBirthDate().equals(oldUserProfile.
                getBirthDate()))
            {
                getChangeRecord("DOB").setOldBirthDate(oldUserProfile.
                    getBirthDate());
                oldUserProfile.setBirthDate(newUserProfile.getBirthDate());
                setDemographicInfoChange();
            }
        }

        private void processSsnChange() {
            if (!StringUtils.equals(newUserProfile.getSsn(), oldUserProfile.
                getSsn()))
            {
                getChangeRecord("SSN").setOldSsn(oldUserProfile.getSsn());
                oldUserProfile.setSsn(newUserProfile.getSsn());
                setDemographicInfoChange();
            }
        }

        private void processGenderChange() {
            if (!equals(newUserProfile.getGender(), oldUserProfile.
                getGender()))
            {
                getChangeRecord("Gender").setOldGender(oldUserProfile.
                    getGender());
                oldUserProfile.setGender(newUserProfile.getGender());
                setDemographicInfoChange();
            }
        }

        @SuppressWarnings("unused")
		private void processAddressChange() {
            processAddressStreet1Change();
            processAddressStreet2Change();
            processAddressCityChange();
            processAddressPostalCodeChange();
            processAddressStateChange();
            processAddressProvinceChange();
            processAddressCountryChange();
        }
        private void processAddressStreet1Change() {
            if (!StringUtils.equals(newUserProfile.getAddressStreet1(),
                oldUserProfile.getAddressStreet1()))
            {
                getChangeRecord("AddressStreet1").setOldAddressStreet1(
                    oldUserProfile.getAddressStreet1());
                oldUserProfile.setAddressStreet1(newUserProfile.
                    getAddressStreet1());
                setDemographicInfoChange();
            }
        }
        private void processAddressStreet2Change() {
            if (!StringUtils.equals(newUserProfile.getAddressStreet2(),
                oldUserProfile.getAddressStreet2()))
            {
                getChangeRecord("AddressStreet2").setOldAddressStreet2(
                    oldUserProfile.getAddressStreet2());
                oldUserProfile.setAddressStreet2(newUserProfile.
                    getAddressStreet2());
                setDemographicInfoChange();
            }
        }
        private void processAddressCityChange() {
            if (!StringUtils.equals(newUserProfile.getAddressCity(),
                oldUserProfile.getAddressCity()))
            {
                getChangeRecord("StreetCity").setOldAddressCity(oldUserProfile.
                    getAddressCity());
                oldUserProfile.setAddressCity(newUserProfile.
                    getAddressCity());
                setDemographicInfoChange();
            }
        }
        private void processAddressPostalCodeChange() {
            if (!StringUtils.equals(newUserProfile.getAddressPostalCode(),
                oldUserProfile.getAddressPostalCode()))
            {
                getChangeRecord("StreetPostalCode").setOldAddressPostalCode(
                    oldUserProfile.getAddressPostalCode());
                oldUserProfile.setAddressPostalCode(newUserProfile.
                    getAddressPostalCode());
                setDemographicInfoChange();
            }
        }
        private void processAddressStateChange() {
            if (!equals(newUserProfile.getAddressState(), oldUserProfile.
                getAddressState()))
            {
                getChangeRecord("AddressState").setOldAddressState(
                    oldUserProfile.getAddressState());
                oldUserProfile.setAddressState(newUserProfile.
                    getAddressState());
                setDemographicInfoChange();
            }
        }
        private void processAddressProvinceChange() {
            if (!StringUtils.equals(newUserProfile.getAddressProvince(),
                oldUserProfile.getAddressProvince()))
            {
                getChangeRecord("AddressProvince").setOldAddressProvince(
                    oldUserProfile.getAddressProvince());
                oldUserProfile.setAddressProvince(newUserProfile.
                    getAddressProvince());
                setDemographicInfoChange();
            }
        }
        private void processAddressCountryChange() {
            if (!equals(newUserProfile.getAddressCountry(), oldUserProfile.
                getAddressCountry()))
            {
                getChangeRecord("StreetCountry").setOldAddressCountry(
                    oldUserProfile.getAddressCountry());
                oldUserProfile.setAddressCountry(newUserProfile.
                    getAddressCountry());
                setDemographicInfoChange();
            }
        }

        private void processContactInfoChange() {
            processHomePhoneChange();
        }

        private void processHomePhoneChange() {

              // Must ignore home phone number until the phone number can
        	  // handle international phone numbers
//            if (!StringUtils.equals(newUserProfile.getContactInfoHomePhone(),
//                oldUserProfile.getContactInfoHomePhone()))
//            {
//                if (LOG.isDebugEnabled()) {
//                    LOG.debug("Ignoring HomePhone change for patient '" +
//                        oldPatient.getUserProfile().getUserName() +
//                        "', because unable to handle international " +
//                        "phone numbers, so not processing home phone " +
//                        "number changes at all");
//                }
//                getChangeRecord("ContactInfoHomePhone").
//                    setOldContactInfoHomePhone(oldUserProfile.
//                    getContactInfoHomePhone());
//                oldUserProfile.setContactInfoHomePhone(newUserProfile.
//                    getContactInfoHomePhone());
//                setDemographicInfoChange();
//            }
        }


        private void processFacilityChanges() {
            Map<String, Facility> oldFacilities = collectStationNumbers(
                oldPatient);
            Map<String, Facility> newFacilities = collectStationNumbers(
                newPatient);
            if (!CollectionUtils.isEqualCollection(newFacilities.keySet(),
                    oldFacilities.keySet()))
            {
                getChangeRecord("Facilities").setOldFacilityCount(oldPatient.
                    getFacilitys().size());
                Set<Facility> facilities = new HashSet<Facility>();
                for (String stationNumber: newFacilities.keySet()) {
                    Facility facility = oldFacilities.get(stationNumber);
                    if (facility == null) {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Adding facility '" + stationNumber
                                + "'");
                        }
                        facility = newFacilities.get(stationNumber);
                    } else {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Keeping facility '" + stationNumber +
                                "' (" + facility.getId() + ")");
                        }
                    }
                    facilities.add(facility);
                }
                if (LOG.isDebugEnabled()) {
                    for (String stationNumber: oldFacilities.keySet()) {
                        Facility facility = newFacilities.get(stationNumber);
                        if (facility == null) {
                            LOG.debug("Removing facilities '" + stationNumber
                                + "'");
                        }
                    }
                }
                oldPatient.setFacilitys(facilities);
            }
        }

        private Map<String, Facility> collectStationNumbers(Patient patient) {
            assert patient != null : "Must provide a patient";
            Map<String, Facility> facilityNames =
                new HashMap<String, Facility>();
            for (Object f: patient.getFacilitys()) {
                Facility facility = (Facility) f;
                facilityNames.put(facility.getName(), facility);
            }
            return facilityNames;
        }

        private boolean equals(Enum newValue, Enum oldValue) {
            return ((newValue == null) && (oldValue == null))
                || ((newValue != null) && newValue.equals(oldValue));
        }
    }

    private InPersonAuthentication saveIPA(InPersonAuthentication ipa,
            AuthenticationStatus status){
        InPersonAuthenticationBO bo = BusinessObjectFactory
            .createInPersonAuthenticationBO();
        ipa = bo.saveAndUpdateStatus(ipa, status);
        return ipa;
    }
    
    public VoidServiceResponse processBatchMviAuthenticate(String auth) {
        VoidServiceResponse response = new VoidServiceResponse();
        List<InPersonAuthenticationBO> bos = InPersonAuthenticationBO.getMVIPendingAuthPatients();
        if(bos != null && !bos.isEmpty()) {
	        Collection<InPersonAuthentication> ipas = InPersonAuthenticationAssembler.getInPersonAuthenticationCollection(bos);
	        for (InPersonAuthentication ipa : ipas) {
	        	Patient patient = ipa.getPatient();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("PatientServiceImpl.batchMviAuthenticate():authenticating patient with ICN:" + patient.getIcn());
                }
                BooleanServiceResponse patientServiceResponse = 
                	ServiceFactory.createMviIntegrationService().mviAuthenticate(patient, patient.getUserProfile(), true, true);
	            if (patientServiceResponse.getBoolean()) {
	                AuthenticationStatus status = InPersonAuthenticationStatusUtils.AUTHENTICATED;
	            	ipa.setMviAuthenticationStatus("OK");
	                ipa = saveIPA(ipa, status);
	            }
			}
        }
        return response;
    }
    
    public VoidServiceResponse processAsyncBatchMviAuthenticate(String auth) {
    	return processBatchMviAuthenticate(auth);
    }
    
    public VoidServiceResponse processBatchMviUnauthenticate(String unauth) {
        VoidServiceResponse response = new VoidServiceResponse();
        List<InPersonAuthentication> bos = InPersonAuthenticationBO.getMVIPendingUnauthPatients();
        if(bos != null && !bos.isEmpty()) {
	        Collection<InPersonAuthentication> ipas = InPersonAuthenticationAssembler.getInPersonAuthenticationCollection(bos);
	        for (InPersonAuthentication ipa : ipas) {
	        	Patient patient = ipa.getPatient();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("PatientServiceImpl.processBatchMviUnauthenticate():unauthenticating patient with ICN:" + patient.getIcn());
                }
                BooleanServiceResponse patientServiceResponse = 
                	ServiceFactory.createMviIntegrationService().mviUnauthenticateUncorrelateForBatch(patient, patient.getUserProfile(), "SYSTEM");
	            if (patientServiceResponse.getBoolean()) {
	                AuthenticationStatus status = InPersonAuthenticationStatusUtils.UNAUTHENTICATED;
    				ipa.setMviAuthenticationStatus("OK");
	                ipa = saveIPA(ipa, status);
	            } else {
	                if (LOG.isDebugEnabled()) {
	                    LOG.debug("PatientServiceImpl.processBatchMviUnauthenticate(): Error unauthenticating patient with ICN:" + patient.getIcn());
	                }
	            }
			}
        }
        return response;
    }
    public VoidServiceResponse processAsyncBatchMviUnauthenticate(String auth) {
    	return processBatchMviUnauthenticate(auth);
    }

	public PatientCollectionServiceResponse getMVIPendingPatients(Boolean isAuth) {
		PatientCollectionServiceResponse response = new PatientCollectionServiceResponse();
		MviProperties properties = MviProperties.getInstance();
		//int numberToRun = properties.getMviAuthBatchNumberToRun();
		//TODO - Replace with the one above.
		int numberToRun = 0;
		if(isAuth) {
	        List<InPersonAuthentication> bos = InPersonAuthenticationBO.getMVIPendingAuthPatients();
	        if(bos != null && !bos.isEmpty()) {
		        Collection<InPersonAuthentication> ipas = InPersonAuthenticationAssembler.getInPersonAuthenticationCollection(bos);
		        int count = 0;
		        for (InPersonAuthentication ipa : ipas) {
		        	if(count < numberToRun) {
			        	Patient patient = ipa.getPatient();
			        	response.addItem(patient);
			        	count++;
		        	}
		        }
	        }
		} else {
			//numberToRun = properties.getMviUnauthBatchNumberToRun();
			//TODO - Replace with the one above.
			numberToRun = 0;
			List<InPersonAuthentication> bos = InPersonAuthenticationBO.getMVIPendingUnauthPatients();
	        if(bos != null && !bos.isEmpty()) {
		        Collection<InPersonAuthentication> ipas = InPersonAuthenticationAssembler.getInPersonAuthenticationCollection(bos);
		        int count = 0;
		        for (InPersonAuthentication ipa : ipas) {
		        	if(count < numberToRun) {
			        	Patient patient = ipa.getPatient();
			        	response.addItem(patient);
		        	}
		        }
	        }
		}
		return response;
	}

	public StringServiceResponse getMVIAutBatchAdmin() {
		MviProperties properties = MviProperties.getInstance();
		StringServiceResponse response = new StringServiceResponse();
		//response.setString(properties.getMviAuthBatchAdmin());
		response.setString("");
		return response;
	}

	public StringServiceResponse getMviAuthBatchIntervalHours() {
		MviProperties properties = MviProperties.getInstance();
		StringServiceResponse response = new StringServiceResponse();
		//response.setString(properties.getMviAuthBatchIntervalHours().toString());
		response.setString("");
		return response;
	}

	public PatientServiceResponse setPendingAuthAndDeletePatientSynchronization(
			Patient patient) {
        InPersonAuthenticationServiceResponse response = 
        	ServiceFactory.createInPersonAuthenticationService().getIPAPatientById(
        		patient.getId());
        InPersonAuthentication ipa = response.getInPersonAuthentication();
        ipa.setMviAuthenticationStatus("PENDING_AUTH");
        AuthenticationStatus status = InPersonAuthenticationStatusUtils.AUTHENTICATED;
        ipa = saveIPA(ipa, status);
        patient.setPatientSynchronizations(null);
        ServiceFactory.createEntityMaintenanceService().save(patient);
		PatientServiceResponse pResponse = new PatientServiceResponse();
		return pResponse;
	}

}